/*

    File: adv.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrface.h"
#include "readpart.h"
#include "fnctdsk.h"
#include "testdisk.h"
#include "fat.h"
#include "ntfs.h"
#include "adv.h"

void interface_adv(t_param_disk *disk_car, const int debug,const int dump_ind)
{
  int quit;
  int offset=0;
  int current_element_num=0;
  int rewrite=1;
  t_list_part *element;
  t_list_part *list_part=read_part(disk_car, debug);
  t_list_part *current_element=list_part;
  ecrit_rapport("\nInterface Advanced\n");
  for(element=list_part;element!=NULL;element=element->next)
  {
    aff_part_rapport(disk_car,element->part);
  }
  do
  {
    static struct MenuItem menuAdv[]=
    {
      {'q',"Quit","Return to main menu"},
      {'b',"Boot","Boot sector recovery"},
      {0,NULL,NULL}
    };
    int opt_B=0;
    int i;
    int command;
    if(rewrite!=0)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
/*     mvwaddstr(stdscr,22,0,msg_PART_TYPE); */
/*     mvwaddstr(stdscr,5,0,msg_CHOOSE_PART_TYPE); */
      mvwaddstr(stdscr,6,0,msg_PART_HEADER2);
      rewrite=0;
    }
    for(i=0,element=list_part;(element!=NULL) && (i<offset);element=element->next,i++);
    for(i=offset;(element!=NULL) && ((i-offset)<INTER_ADV);i++,element=element->next)
    {
      wmove(stdscr,5+2+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(element==current_element)
      {
	wstandout(stdscr);
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
	wstandend(stdscr);
      } else
      {
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
      }
    }
    if(current_element!=NULL)
    {
      if(is_fat(current_element->part->part_type) ||
	  is_ntfs(current_element->part->part_type))
	opt_B=1;
    }
    quit=0;
    command = wmenuSelect(stdscr,INTER_ADV_Y, INTER_ADV_X, menuAdv, 8,
	opt_B!=0?"bq":"q", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, opt_B);
    switch(command)
    {
      case MENU_UP:
	if(current_element!=NULL)
	{
	  if(current_element->prev!=NULL)
	  {
	    current_element=current_element->prev;
	    current_element_num--;
	  }
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case MENU_DOWN:
	if(current_element!=NULL)
	{
	  if(current_element->next!=NULL)
	  {
	    current_element=current_element->next;
	    current_element_num++;
	  }
	  if(current_element_num>=offset+INTER_ADV)
	    offset++;
	}
	break;
      case 'q':
      case 'Q':
	quit=1;
	break;
      case 'b':
      case 'B':
	if(current_element!=NULL)
	{
	  if(is_fat32(current_element->part->part_type))
	  {
	    fat32_boot_sector(disk_car, current_element->part, debug, dump_ind);
	    rewrite=1;
	  }
	  else if(is_fat(current_element->part->part_type))
	  {
	    fat1x_boot_sector(disk_car, current_element->part, debug, dump_ind);
	    rewrite=1;
	  }
	  else if(is_ntfs(current_element->part->part_type))
	  {
	    ntfs_boot_sector(disk_car, current_element->part, debug, dump_ind);
	    rewrite=1;
	  }
	}
	break;
    }
  } while(quit==0);
  delete_list_part(list_part);
}

int fat1x_boot_sector(t_param_disk *disk_car, t_diskext *partition, const int debug, const int dump_ind)
{
  t_sector buffer_bs;
  const char *options="DR";
  int rescan=1;
  while(1)
  {
    aff_buffer(BUFFER_RESET,"Q");
    if(rescan==1)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      wmove(stdscr,5,0);
      aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
      ecrit_rapport("\nfat1x_boot_sector\n");
      aff_part_rapport(disk_car,partition);
      aff_buffer(BUFFER_ADD,"Boot sector\n");
      if(disk_car->read(disk_car,1, &buffer_bs, partition->lba)!=0)
      {
	ecrit_rapport(msg_CHKFAT_RERR); return 1;
      }
      if(test_FAT(disk_car,(const struct fat_boot_sector *)buffer_bs,partition,debug,0)==0)
      {
	aff_buffer(BUFFER_ADD,"OK\n");
      }
      rescan=0;
    }
    /*  mvwaddstr(stdscr,23,0,msg_PART_TYPE); */
    switch(aff_buffer(BUFFER_DISPLAY,options,stdscr))
    {
      case 0:
	return 0;
      case 3: /* R : rebuild boot sector */
	rebuild_FAT_BS(disk_car,partition,debug,dump_ind,1);
	rescan=1;
	break;
      case 4:
	{
	  WINDOW *window=newwin(0,0,0,0);	/* full screen */
	  keypad(window, TRUE); /* Need it to get arrow key */
	  aff_copy(window);
	  wmove(window,4,0);
	  wdoprintf(window,"%s",disk_car->description(disk_car));
	  wmove(window,5,0);
	  aff_part(window,AFF_PART_ORDER,disk_car,partition);
	  ecrit_rapport("Boot sector\n");
	  mvwaddstr(window,6,0, "Boot sector");
	  dump(window,buffer_bs,SECTOR_SIZE);
	  delwin(window);
#ifdef DJGPP
	  wredrawln(stdscr,0,stdscr->_maxy);	/* redrawwin def is boggus in pdcur24 */
#else
	  redrawwin(stdscr);	/* stdscr has been corrupted by window */
#endif
	}
	break;
    }
  }
}

int fat32_boot_sector(t_param_disk *disk_car, t_diskext *partition, const int debug, const int dump_ind)
{
  unsigned char buffer_bs[3*SECTOR_SIZE];
  unsigned char buffer_backup_bs[3*SECTOR_SIZE];
  const char *options="";
  int rescan=1;
  while(1)
  {
    aff_buffer(BUFFER_RESET,"Q");
    if(rescan==1)
    {
      int opt_over=0;
      int opt_B=0;
      int opt_O=0;
      options="DR";
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      wmove(stdscr,5,0);
      aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
      ecrit_rapport("\nfat32_boot_sector\n");
      aff_part_rapport(disk_car,partition);
      aff_buffer(BUFFER_ADD,"Boot sector\n");
      if(disk_car->read(disk_car,3, &buffer_bs, partition->lba)!=0)
      {
	ecrit_rapport(msg_CHKFAT_RERR); return 1;
      }
      if(test_FAT(disk_car,(struct fat_boot_sector *)buffer_bs,partition,debug,0)==0)
      {
	aff_buffer(BUFFER_ADD,"OK\n");
	opt_O=1;
	opt_over=1;
      }
      aff_buffer(BUFFER_ADD,"\nBackup boot sector\n");
      if(disk_car->read(disk_car,3, &buffer_backup_bs, partition->lba+6)!=0)
      {
	ecrit_rapport(msg_CHKFAT_RERR); return 1;
      }
      if(test_FAT(disk_car,(struct fat_boot_sector *)buffer_backup_bs,partition,debug,0)==0)
      {
	aff_buffer(BUFFER_ADD,"OK\n");
	opt_B=1;
	opt_over=1;
      }
      aff_buffer(BUFFER_ADD,"\n");
      if((memcmp(buffer_bs,buffer_backup_bs,0x3E8)==0)&&(memcmp(&buffer_bs[0x3F0],&buffer_backup_bs[0x3F0],0x600-0x3F0))==0)
      {
	aff_buffer(BUFFER_ADD,"Sectors are identical.\n");
	opt_over=0;
      }
      else
      {
	if(memcmp(buffer_bs,buffer_backup_bs,SECTOR_SIZE)!=0)
	  aff_buffer(BUFFER_ADD,"First sectors (Boot code and partition information) are not identical.\n");
	if((memcmp(&buffer_bs[SECTOR_SIZE],&buffer_backup_bs[SECTOR_SIZE],0x1E8)!=0)||
	    (memcmp(&buffer_bs[SECTOR_SIZE+0x1F0],&buffer_backup_bs[SECTOR_SIZE+0x1F0],SECTOR_SIZE-0x1F0)!=0))
	  aff_buffer(BUFFER_ADD,"Second sectors (cluster information) are not identical.\n");
	if(memcmp(&buffer_bs[2*SECTOR_SIZE],&buffer_backup_bs[2*SECTOR_SIZE],SECTOR_SIZE)!=0)
	  aff_buffer(BUFFER_ADD,"Third sectors (Second part of boot code) are not identical.\n");
      }

      if(opt_over!=0)
      {
	if(opt_B!=0 && opt_O!=0)
	  options="DOBR";
	else if(opt_B!=0)
	  options="DBR";
	else if(opt_O!=0)
	  options="DOR";
      }
      rescan=0;
    }
    switch(aff_buffer(BUFFER_DISPLAY,options,stdscr))
    {
      case 0:
	return 0;
      case 1: /* O : copy original boot sector over backup boot */
	if(ask_confirmation("Copy original FAT32 boot sector over backup boot, confirm ? (Y/N)")!=0)
	{
	  ecrit_rapport("copy original boot sector over backup boot\n");
	  if(disk_car->write(disk_car,3, &buffer_bs, partition->lba+6)!=0)
	  {
	    display_message("Write error: Can't overwrite FAT32 backup boot sector\n");
	  }
	  rescan=1;
	}
	break;
      case 2: /* B : copy backup boot sector over boot sector */
	if(ask_confirmation("Copy backup FAT32 boot sector over boot sector, confirm ? (Y/N)")!=0)
	{
	  ecrit_rapport("copy backup boot sector over boot sector\n");
	  if(disk_car->write(disk_car,3, &buffer_backup_bs, partition->lba)!=0)
	  {
	    display_message("Write error: Can't overwrite FAT32 boot sector\n");
	  }
	  rescan=1;
	}
	break;
      case 3: /* R : rebuild boot sector */
	rebuild_FAT_BS(disk_car,partition,debug,dump_ind,1);
	rescan=1;
	break;
      case 4:
	{
	  WINDOW *window=newwin(0,0,0,0);	/* full screen */
	  keypad(window, TRUE); /* Need it to get arrow key */
	  aff_copy(window);
	  wmove(window,4,0);
	  wdoprintf(window,"%s",disk_car->description(disk_car));
	  wmove(window,5,0);
	  aff_part(window,AFF_PART_ORDER,disk_car,partition);
	  ecrit_rapport("Boot sector                        Backup boot sector\n");
	  mvwaddstr(window,6,0, "Boot sector                        Backup boot sector");
	  dump2(window,buffer_bs,buffer_backup_bs,3*SECTOR_SIZE);
	  delwin(window);
#ifdef DJGPP
	  wredrawln(stdscr,0,stdscr->_maxy);	/* redrawwin def is boggus in pdcur24 */
#else
	  redrawwin(stdscr);	/* stdscr has been corrupted by window */
#endif
	  dump_2fat_rapport((const struct fat_boot_sector*)&buffer_bs,(const struct fat_boot_sector*)&buffer_backup_bs,UP_FAT32);

	}
	break;
    }
  }
}

int ntfs_boot_sector(t_param_disk *disk_car, t_diskext *partition, const int debug, const int dump_ind)
{
  unsigned char buffer_bs[SECTOR_SIZE];
  unsigned char buffer_backup_bs[SECTOR_SIZE];
  const char *options="";
  int rescan=1;
  while(1)
  {
    aff_buffer(BUFFER_RESET,"Q");
    if(rescan==1)
    {
      int opt_over=0;
      int opt_B=0;
      int opt_O=0;
      options="D";
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      wmove(stdscr,5,0);
      aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
      ecrit_rapport("\nntfs_boot_sector\n");
      aff_part_rapport(disk_car,partition);
      aff_buffer(BUFFER_ADD,"Boot sector\n");
      if(disk_car->read(disk_car,1, &buffer_bs, partition->lba)!=0)
      {
	ecrit_rapport(msg_CHKFAT_RERR); return 1;
      }
      if(test_NTFS(disk_car,(struct ntfs_boot_sector*)buffer_bs,partition,debug,0)!=0)
      {
	aff_buffer(BUFFER_ADD,"OK\n");
	opt_O=1;
	opt_over=1;
      }
      aff_buffer(BUFFER_ADD,"\nBackup boot sector\n");
      if(disk_car->read(disk_car,1, &buffer_backup_bs, partition->lba+partition->part_size-1)!=0)
      {
	ecrit_rapport(msg_CHKFAT_RERR); return 1;
      }
      if(test_NTFS(disk_car,(struct ntfs_boot_sector*)buffer_backup_bs,partition,debug,0)==0)
      {
	aff_buffer(BUFFER_ADD,"OK\n");
	opt_B=1;
	opt_over=1;
      }
      aff_buffer(BUFFER_ADD,"\n");
      if(memcmp(buffer_bs,buffer_backup_bs,SECTOR_SIZE)==0)
      {
	dump_ntfs_rapport((const struct ntfs_boot_sector *)buffer_bs);
	aff_buffer(BUFFER_ADD,"Sectors are identical.\n");
	opt_over=0;
      }
      else
      {
	dump_2ntfs_rapport((const struct ntfs_boot_sector *)buffer_bs, (const struct ntfs_boot_sector *)buffer_backup_bs);
	aff_buffer(BUFFER_ADD,"Sectors are not identical.\n");
      }
      if(opt_over!=0)
      {
	if(opt_B!=0 && opt_O!=0)
	  options="DOB";
	else if(opt_B!=0)
	  options="DB";
	else if(opt_O!=0)
	  options="DO";
      }
      rescan=0;
    }
    switch(aff_buffer(BUFFER_DISPLAY,options,stdscr))
    {
      case 0:
	return 0;
      case 1: /* O : copy original boot sector over backup boot */
	if(ask_confirmation("Copy original NTFS boot sector over backup boot, confirm ? (Y/N)")!=0)
	{
	  ecrit_rapport("copy original boot sector over backup boot\n");
	  if(disk_car->write(disk_car,3, &buffer_bs, partition->lba+6)!=0)
	  {
	    display_message("Write error: Can't overwrite NTFS backup boot sector\n");
	  }
	  rescan=1;
	}
	break;
      case 2: /* B : copy backup boot sector over boot sector */
	if(ask_confirmation("Copy backup NTFS boot sector over boot sector, confirm ? (Y/N)")!=0)
	{
	  ecrit_rapport("copy backup boot sector over boot sector\n");
	  if(disk_car->write(disk_car,3, &buffer_backup_bs, partition->lba)!=0)
	  {
	    display_message("Write error: Can't overwrite NTFS boot sector\n");
	  }
	  rescan=1;
	}
	break;
      case 3: /* R : rebuild boot sector */
	rebuild_NTFS_BS(disk_car,partition,debug,dump_ind,1);
	rescan=1;
	break;
      case 4:
	{
	  WINDOW *window=newwin(0,0,0,0);	/* full screen */
	  keypad(window, TRUE); /* Need it to get arrow key */
	  aff_copy(window);
	  wmove(window,4,0);
	  wdoprintf(window,"%s",disk_car->description(disk_car));
	  wmove(window,5,0);
	  aff_part(window,AFF_PART_ORDER,disk_car,partition);
	  ecrit_rapport("Boot sector                        Backup boot sector\n");
	  mvwaddstr(window,6,0, "Boot sector                        Backup boot sector");
	  dump2(window,buffer_bs,buffer_backup_bs,SECTOR_SIZE);
	  delwin(window);
#ifdef DJGPP
	  wredrawln(stdscr,0,stdscr->_maxy);	/* redrawwin def is boggus in pdcur24 */
#else
	  redrawwin(stdscr);	/* stdscr has been corrupted by window */
#endif
	}
	break;
    }
  }
}

